/*
	DigitalFlipbook Source Code License Agreement
	Copyright(c) 2008 Mark Walters. All rights reserved.
		
	Please read this Source Code License Agreement carefully before using
	the source code.
		
	Mark Walters grants to you a perpetual, worldwide, non-exclusive,
	no-charge, royalty-free, irrevocable copyright license, to reproduce,
	prepare derivative works of, publicly display, publicly perform, and
	distribute this source code and such derivative works in source or
	object code form without any attribution requirements.
	
	The names "Mark Walters" and "DigitalFlipbook" must not be used to endorse or promote products
	derived from the source code without prior written permission.
		
	You agree to indemnify, hold harmless and defend Mark Walters from and
	against any loss, damage, claims or lawsuits, including attorney's
	fees that arise or result from your use or distribution of the source
	code.
		
	THIS SOURCE CODE IS PROVIDED "AS IS" AND "WITH ALL FAULTS", WITHOUT
	ANY TECHNICAL SUPPORT OR ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING,
	BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. ALSO, THERE IS NO WARRANTY OF
	NON-INFRINGEMENT, TITLE OR QUIET ENJOYMENT. IN NO EVENT SHALL MARK WALTERS
	OR HIS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
	EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
	PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
	OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
	WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
	OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOURCE CODE, EVEN IF
	ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
package com.digitalflipbook.geoglobe
{
	import caurina.transitions.Tweener;
	
	import flash.events.Event;
	import flash.events.MouseEvent;
	
	import org.papervision3d.core.math.Number3D;
	import org.papervision3d.events.InteractiveScene3DEvent;
	import org.papervision3d.objects.DisplayObject3D;
	import org.papervision3d.view.BasicView;

	public class GlobeView extends BasicView
	{
		protected var latitudeDegreeOffset:Number = 90;
		protected var longitudeDegreeOffset:Number = 15;
		
		protected var cameraTarget:DisplayObject3D;
		protected var globe:Globe;
		protected var globeMouseDiffX:Number = 0;
		protected var globeMouseDiffY:Number = 0;
		protected var mouseDown:Boolean = false;
		protected var geocoder:Geocoder;
		
		public function GlobeView( viewportWidth:Number=640, viewportHeight:Number=320, scaleToStage:Boolean=true, interactive:Boolean=false, cameraType:String="FREECAMERA3D" )
		{
			//Setup the basic view.
			super( viewportWidth, viewportHeight, scaleToStage, interactive, cameraType );
			init();
			startRendering();
		}
		
		public function addMarker( location:String ):void
		{
			geocoder.locate( location );
		}
		
		public function clearMarkers():void
		{
			for each( var obj:DisplayObject3D in globe.children )
			{
				if( obj is Marker ) globe.removeChild( obj );
			}
		}
		
		protected function init():void
		{
			//Create a null object for the camera to copy.
			cameraTarget = new DisplayObject3D();
			scene.addChild( cameraTarget );
			
			//Rotate the camera target to have the camera face America.
			cameraTarget.yaw( 180 );
			
			//Create the globe.
			globe = new Globe();
			scene.addChild( globe );
			
			//Set the camera's focus and zoom.
			camera.focus = 1100;
			camera.zoom = 1;
			
			//Listen for mouse events.
			addEventListener( MouseEvent.MOUSE_DOWN, mouseDownHandler );
			addEventListener( MouseEvent.MOUSE_UP, mouseUpHandler );
			addEventListener( MouseEvent.ROLL_OUT, mouseUpHandler );
			
			//Initialize the geo geocoder.
			geocoder = new Geocoder();
			geocoder.addEventListener( GeocoderEvent.GEOCODER_SUCCESS, geocoderSuccessHandler );
			geocoder.addEventListener( GeocoderEvent.GEOCODER_FAILURE, geocoderFailureHandler );
			geocoder.addEventListener( GeocoderEvent.GEOCODER_UNAVAILABLE, geocoderUnavailableHandler );
		}
		
		protected function placeMarker( marker:Marker, latitude:Number, longitude:Number ):void
		{
			//Translate the geo coordinates to 3D coordinates.
			var markerVector:Number3D = translateGeoCoords( latitude, longitude, globe.radius );
			
			marker.x = markerVector.x;
			marker.y = markerVector.y;
			marker.z = markerVector.z;
			
			//Align the marker with the globe's surface.
			marker.lookAt( DisplayObject3D.ZERO );
			
			globe.addChild( marker );
			
			marker.addEventListener( InteractiveScene3DEvent.OBJECT_CLICK, markerClickHandler );
		}
		
		private function translateGeoCoords( latitude:Number, longitude:Number, radius:Number ):Number3D
		{
			//Convert latitude and longitude to radians.
			latitude = Math.PI * latitude / 180;
			longitude = Math.PI * longitude / 180;
			
			//Adjust latitude and longitude by radians.
			latitude -= ( latitudeDegreeOffset * ( Math.PI/180 ) ); // offset latitude by n degrees (in radians).
			longitude -= ( longitudeDegreeOffset * ( Math.PI/180 ) ); // offset longitude by n degrees (in radians).
			
			//Spherical to cartesian coordinate conversion formula.
			
			//----		r (radius)				----//
			//----		θ (theta) = latitude	----//
			//----		φ (phi) = longitude		----//
			
			//----		x = r sin θ cos φ		----//
			//----		y = r sin θ sin φ		----//
			//----		z = r cos θ				----//
			
			var x:Number = radius * Math.sin( latitude ) * Math.cos( longitude );
			var y:Number = radius * Math.sin( latitude ) * Math.sin( longitude );
			var z:Number = radius * Math.cos( latitude );
			
			//Switch z and y (since z is forward) (see the right-hand rule).
			return new Number3D( x, z, y );
		}
		
		protected override function onRenderTick( event:Event=null ):void
		{
			if( mouseDown )
			{
				//Set the target rotation properties for the camera based on the mouse position.
				var rotationY:Number = -( -mouseX - globeMouseDiffX );
				var rotationX:Number = -( mouseY - globeMouseDiffY );
				
				//Set a maximum amount that the camera can rotate around the x-axis.
				rotationX = rotationX > 10 ? 10 : rotationX < -30 ? -30 : rotationX;
				
				//Tween the null object.
				Tweener.addTween( cameraTarget, { rotationY:rotationY, rotationX:rotationX, time:3 } );
			}
			
			//Copy the null object's transform into the camera.
			camera.copyTransform( cameraTarget );
			
			//Reposition the camera away from the globe and null object.
			camera.moveBackward( 360 );
			
			camera.moveDown( 60 );
			
			//Render as usual
			super.onRenderTick( event );
		}
		
		protected function mouseDownHandler( evt:MouseEvent ):void
		{
			//Track the amount the mouse has moved.
			globeMouseDiffX = -mouseX + cameraTarget.rotationY;
			globeMouseDiffY = mouseY + cameraTarget.rotationX;
			mouseDown = true;
		}
		
		protected function mouseUpHandler( evt:MouseEvent ):void
		{
			mouseDown = false;
		}
		
		protected function geocoderSuccessHandler( evt:GeocoderEvent ):void
		{
			trace( evt.location );
			trace( evt.latitude );
			trace( evt.longitude );
			
			var marker:Marker = new Marker();
			placeMarker( marker, evt.latitude, evt.longitude );
			//changeView( marker );
		}
		
		protected function changeView( obj:DisplayObject3D ):void
		{
			Tweener.removeAllTweens();
			Tweener.addTween( cameraTarget, { rotationX:obj.rotationY, rotationY:obj.rotationX, time:3 } );
		}
		
		protected function geocoderFailureHandler( evt:GeocoderEvent ):void
		{
			trace( "GEOCODER FAILURE" );
		}
		
		protected function geocoderUnavailableHandler( evt:GeocoderEvent ):void
		{
			trace( "GEOCODER UNAVAILABLE" );
		}
		
		protected function markerClickHandler( evt:InteractiveScene3DEvent ):void
		{
			
		}
		
	}
}