# ****************************************************************************** #
# +----------------------------------------------------------------------------+ #
# | File : Publish.ps1    Updated 10/31                                                     | #
# +----------------------------------------------------------------------------+ #
# ****************************************************************************** #

<#
.SYNOPSIS
Publish time script to perform the necessary tasks on the client machine before publishing App-V Adobe Acrobat Reader DC.

.DESCRIPTION
.\Publish.ps1 [-CreateLog] [-LogFile <path to logfile>] [-ChannelName {"DC|2015|2017"}]

.EXAMPLE
.\Publish.ps1 -ChannelName "2015|2017"
Runs Publish.ps1 for Classic track.

.EXAMPLE
.\Publish.ps1 -CreateLog -LogFile testing.log -ChannelName "DC"
Runs Publish.ps1 for Continuous track and creates log inside testing.log file.
#>

param (
#Identifies the delivery track of the application, should match the track from which package is created. Value is either "DC" or "2015" or "2017".
[String]$ChannelName="2015",
#Specifies the filepath to which script logs will be written.
[String]$LogFile="$env:Temp\Reader-Appv.log",
#Switched parameter that generates Script Logs on the client machine, at the location described by -LogFile parameter.
[Switch]$CreateLog)
# +---------------------------------------------------------------------------
# | FUNCTIONS
# |
# | description of function, parameters and what it returns (if any)                                                                                        
# +----------------------------------------------------------------------------
 
function ShowHeading {
##-----------------------------------------------------------------------
## Function: ShowHeading
## Purpose: Use to write a header for the script
##-----------------------------------------------------------------------
        Log "---------------------------------------------------------" 
        Log "-                       Publish.ps1                     -"
        Log "-       Post Publishing Script for Reader App-V        -"
        Log "---------------------------------------------------------"
}
 
function Log {
##-----------------------------------------------------------------------
## Function: Log
## Purpose: Used to write output to a log file
##-----------------------------------------------------------------------
    param ([string]$argOut)
    $kDateTime = get-date -uFormat "%d-%m-%Y %H:%M"
    $kOut = ($kDateTime+ ": " + $argOut)
    If ($LogFile -and $CreateLog) {"$kOut" | out-file -filepath $LogFile -append}
}
 
function PrintAndLog {
##-----------------------------------------------------------------------
## Function: PrintAndLog
## Purpose: Used to write output to the screen
##-----------------------------------------------------------------------
    param ([string]$argOut)
            write-host $argOut
    If ($CreateLog) {Log $argOut}
}
 
function SuppressUpdater
{
##------------------------------------------------------------------------------------------
## Function: SuppressUpdater
## Purpose: This function is responsible for Suppressing Updater in App-V Adobe Acrobat Reader DC
##------------------------------------------------------------------------------------------
    PrintAndLog "-------------------------------------------------------------------------------"
    PrintAndLog "Disabling Updater for Adobe Acrobat Reader DC, so that it does not update through ARM"
    PrintAndLog "-------------------------------------------------------------------------------"
             
    # Registries to be made in the native hive
    $kReaderPoliciesHive = "HKLM:\SOFTWARE\Policies\Adobe\Acrobat Reader\$ChannelName"
    $kReaderLockDownHive = $kReaderPoliciesHive + "\FeatureLockDown"
    If(Test-Path $kReaderLockDownHive) {
        PrintAndLog "DEBUG: [Native] - FeatureLockDown Key available"
    } Else {
        New-Item -Path $kReaderPoliciesHive -Name FeatureLockDown -Force
        PrintAndLog "DEBUG: [Native] - FeatureLockDown Key created"
    }
    Set-ItemProperty -Path $kReaderLockDownHive -Name "bUpdater" -Value 0  # Disabling Updating of the product.
         
    If([System.Environment]::Is64BitProcess) {
        # + Registries to be made in the Wow6432 Hive
        $kReaderPoliciesWow6432Hive = "HKLM:\SOFTWARE\Wow6432Node\Policies\Adobe\Acrobat Reader\$ChannelName"
        $kReaderLockDownWow6432Hive = $kReaderPoliciesHive + "\FeatureLockDown"
        If(Test-Path $kReaderLockDownWow6432Hive) {
            PrintAndLog "DEBUG: [Wow6432] - FeatureLockDown Key available"
        } Else {
            New-Item -Path $kReaderPoliciesWow6432Hive -Name FeatureLockDown -Force
            PrintAndLog "DEBUG: [Wow6432] - FeatureLockDown Key created"
        }
        Set-ItemProperty -Path $kReaderLockDownWow6432Hive -Name "bUpdater" -Value 0  # Disabling Updating of the product.
        PrintAndLog "DEBUG: [Wow6432] - bUpdater Key created in FeatureLockDown"
    }
    PrintAndLog "DEBUG: [Native] - bUpdater Key created in FeatureLockDown"
}

function SuppressNotifications
{
##---------------------------------------------------------------------------------------
## Function: SuppressNotifications
## Purpose: This function is responsible for Suppressing Notifications in App-V Acrobat Reader
##---------------------------------------------------------------------------------------
    PrintAndLog "-------------------------------------------------------------------------------"
    PrintAndLog "Disabling Notifications for Reader, as it is not supported in AppV"
    PrintAndLog "-------------------------------------------------------------------------------"
            
    # Registries to be made in the native hive
    $kReaderPoliciesHive = "HKLM:\SOFTWARE\Policies\Adobe\Acrobat Reader\$ChannelName"
    $kReaderLockDownHive = $kReaderPoliciesHive + "\FeatureLockDown"
    If(Test-Path $kReaderLockDownHive) {
        PrintAndLog "DEBUG: [Native] - FeatureLockDown Key available"
    } Else {
        New-Item -Path $kReaderPoliciesHive -Name FeatureLockDown -Force
    }
    $kReaderServicesLockDownHive = $kReaderLockDownHive + "\cServices"
    If(Test-Path $kReaderServicesLockDownHive) {
        PrintAndLog "DEBUG: [Native] - Services Key available"
    } Else {
        New-Item -Path $kReaderLockDownHive -Name cServices -Force
    }
    Set-ItemProperty -Path $kReaderServicesLockDownHive -Name "bToggleNotifications" -Value 1  # Disabling Notifications of the product.
        
    If([System.Environment]::Is64BitProcess) {
        # + Registries to be made in the Wow6432 Hive
        $kReaderPoliciesWow6432Hive = "HKLM:\SOFTWARE\Wow6432Node\Policies\Adobe\Acrobat Reader\$ChannelName"
        $kReaderLockDownWow6432Hive = $kReaderPoliciesWow6432Hive + "\FeatureLockDown"
        If(Test-Path $kReaderLockDownWow6432Hive) {
            PrintAndLog "DEBUG: [Native] - FeatureLockDown Key available"
        } Else {
            New-Item -Path $kReaderPoliciesWow6432Hive -Name FeatureLockDown -Force
        }
        $kReaderServicesLockDownWow6432Hive = $kReaderLockDownWow6432Hive + "\cServices"
        If(Test-Path $kReaderServicesLockDownWow6432Hive) {
            PrintAndLog "DEBUG: [Native] - Services Key available"
        } Else {
            New-Item -Path $kReaderLockDownWow6432Hive -Name cServices -Force
        }
        Set-ItemProperty -Path $kReaderServicesLockDownWow6432Hive -Name "bToggleNotifications" -Value 1  # Disabling Notifications of the product.
    }
}
 
function AppendToPassThrough
{
    PrintAndLog "-------------------------------------------------------------------------------"
    PrintAndLog "Adding the necessary entries in PassThroughPaths key of App-V"
    PrintAndLog "-------------------------------------------------------------------------------"
     
    $kPassThroughHive = "HKLM:\SOFTWARE\Microsoft\AppV\Subsystem\VirtualRegistry"
    $kPassThroughKey = "PassThroughPaths"
    $arrPassThroughEntriesPresent = (Get-ItemProperty $kPassThroughHive $kPassThroughKey).$kPassThroughKey
    $arrPassThroughEntriesToAdd = "HKEY_CURRENT_USER\Software\Adobe\Acrobat Reader", "HKEY_CURRENT_USER\Software\Adobe\Adobe Acrobat\$ChannelName\DiskCabs", "HKEY_CURRENT_USER\Software\Adobe\Adobe Synchronizer", "HKEY_CURRENT_USER\Software\Adobe\CommonFiles", "HKEY_CURRENT_USER\Software\Microsoft\Speech", "HKEY_CURRENT_USER\Software\Lotus\Notes"
     
    PrintAndLog "`nDEBUG: The entries available presently in passthrough are" #$arrPassThroughEntriesPresent"
    PrintAndLog $arrPassThroughEntriesPresent
    PrintAndLog "`nDEBUG: The entries to be added in passthrough are" #$arrPassThroughEntriesToAdd"
    PrintAndLog $arrPassThroughEntriesToAdd

    ForEach($Entry in $arrPassThroughEntriesToAdd){
        if($arrPassThroughEntriesPresent -notcontains $Entry){
            $arrPassThroughEntriesPresent += $Entry
        }
    }
    PrintAndLog "`nDEBUG: The entries to be present in passthrough are" #$arrPassThroughEntriesPresent"
    PrintAndLog $arrPassThroughEntriesPresent
    Set-ItemProperty $kPassThroughHive $kPassThroughKey -value $arrPassThroughEntriesPresent -type MultiString
    PrintAndLog "`nDEBUG: The keys successfully added to App-V PassThroughPaths"
}   
 
function CreateAppVRegistry
{
##---------------------------------------------------------------------------------
## Function: CreateAppVRegistry
## Purpose: This function is used to create appv key
##---------------------------------------------------------------------------------
    PrintAndLog "-------------------------------------------------------------------------------"
    PrintAndLog "Setting appv registry key"
    PrintAndLog "-------------------------------------------------------------------------------"
    # Registries to be made in Native Hive
    $kAcrobatHive = "HKLM:\Software\Adobe\Acrobat Reader\$ChannelName"
    $kAcrobatAppvHive = $kAcrobatHive + "\appv"
    If (Test-Path $kAcrobatAppvHive){
        PrintAndLog "DEBUG: [Native] - $kAcrobatAppvHive is available"
    } Else {
        New-Item -Path $kAcrobatHive -Name appv -Force
    }
            
    # + Registries to be made in Wow Hive
    If([System.Environment]::Is64BitProcess){
        $kAcrobatWow64Hive = "HKLM:\Software\Wow6432Node\Adobe\Acrobat Reader\$ChannelName"
        $kAcrobatWow64AppvHive = $kAcrobatWow64Hive + "\appv"
        If (Test-Path $kAcrobatWow64AppvHive) {
            PrintAndLog "DEBUG: [Wow6432] - $kAcrobatWow64AppvHive is available"
        } Else {
            New-Item -Path $kAcrobatWow64Hive -Name appv -Force
        }        
   }
}
 
function Usage
{
##-----------------------------------------------------------------------
## Function: Usage
## Purpose: This function is used to describe how to use this script
##-----------------------------------------------------------------------
PrintAndLog
@"
    Usage:
         
     .\Publish.ps1 [-CreateLog] [[-LogFile <path to logfile>] [-ChannelName {"DC|2015|2017"}]
     Runs the Publish time script to perform the necessary tasks on the client machine before publishing App-V Adobe Acrobat Reader DC.
     e.g. .\Publish.ps1 -CreateLog -LogFile testing.log -ChannelName "2015|2017"
         
"@
    return
}
 
# +---------------------------------------------------------------------------
# | MAIN ROUTINE                                                                                        
# +----------------------------------------------------------------------------
function Main
{
    If (Test-Path $LogFile) {Remove-Item -path $LogFile}
    ShowHeading
    SuppressUpdater
    SuppressNotifications
    AppendToPassThrough
    CreateAppVRegistry   
    PrintAndLog "-------------------------------------------------------------------------------"
}
Main