# ****************************************************************************** #
# +----------------------------------------------------------------------------+ #
# | File : Unpublish.ps1       Updated 10/31                                                | #
# +----------------------------------------------------------------------------+ #
# ****************************************************************************** #

<#
.SYNOPSIS
Unpublish time script to remove the dependencies of Adobe Acrobat Reader DC created as a part of publishing the package to the machine.

.DESCRIPTION
.\Unpublish.ps1 [-CreateLog] [-LogFile <path to logfile>] [-ChannelName {"DC|2015|2017"}]

.EXAMPLE
.\Unpublish.ps1 -ChannelName "2015|2017"
Runs Unpublish.ps1 for Classic track.

.EXAMPLE
.\Unpublish.ps1 -CreateLog -LogFile testing.log -ChannelName "DC"
Runs Unpublish.ps1 for Continuous track and creates log inside testing.log file.
#>

param(
#Identifies the delivery track of the application, should match the track from which package is created. Value is either "DC" or "2015" or "2017".
[string]$ChannelName="2015",
#Switched parameter that generates Script Logs on the client machine, at the location described by -LogFile parameter.
[switch]$CreateLog,
#Specifies the filepath to which script logs will be written.
[string]$LogFile="$env:Temp\Reader-Appv.log")
# +----------------------------------------------------------------------------
# | FUNCTIONS
# |
# | description of function, parameters and what it returns (if any)                                                                                        
# +----------------------------------------------------------------------------
 
function ShowHeading {
##-----------------------------------------------------------------------
## Function: ShowHeading
## Purpose: Use to write a header for the script
##-----------------------------------------------------------------------
        Log "---------------------------------------------------------" 
        Log "-                      Unpublish.ps1                    -"
        Log "-   Unpublish Time Script for Adobe Acrobat Reader DC App-V   -"
        Log "---------------------------------------------------------"
}
 
function Log {
##-----------------------------------------------------------------------
## Function: Log
## Purpose: Used to write output to a log file
##-----------------------------------------------------------------------
    param ([string]$argOut)
    $kDateAndTime = get-date -uFormat "%d-%m-%Y %H:%M"
    $kOut = ($kDateAndTime+ ": " + $argOut)
    If ($LogFile -and $CreateLog) { "$kOut" | out-file -filepath $LogFile -append }
}
 
function PrintAndLog {
##-----------------------------------------------------------------------
## Function: PrintAndLog
## Purpose: Used to write output to the screen and Log it to a File
##-----------------------------------------------------------------------
    param ([string]$argOut)
    write-host $argOut
    If ($CreateLog) { Log $argOut }
}
 
function RemoveUpdaterKey
{
##---------------------------------------------------------------------------------
## Function: RemoveUpdaterKey
## Purpose: This function removes the Updater key in App-V Adobe Acrobat Reader DC
##---------------------------------------------------------------------------------
    PrintAndLog "-------------------------------------------------------------------------------"
    PrintAndLog "Removing Policies key to shutdown bUpdater"
    PrintAndLog "-------------------------------------------------------------------------------"
             
    # Registries that are made in the native hive
    $kReaderLockDownHive = "HKLM:\Software\Policies\Adobe\Acrobat Reader\$ChannelName\FeatureLockDown"
    If(Test-Path $kReaderLockDownHive) {
        PrintAndLog "DEBUG: [Native] - FeatureLockDown Key available"
        $objReaderKey = Get-Item -LiteralPath $kReaderLockDownHive
        If ($objReaderKey.GetValue("bUpdater", $null) -ne $null) {
            PrintAndLog "DEBUG: [Native] - bUpdater Key present"
            Remove-ItemProperty -Path $kReaderLockDownHive -Name bUpdater -Force
            PrintAndLog "DEBUG: [Native] - bUpdater Key removed"
        } Else {
            PrintAndLog "DEBUG: [Native] - bUpdater Key not present"
        }
    } Else {
        PrintAndLog "DEBUG: [Native] - FeatureLockDown Key not present"
    }
         
    If([System.Environment]::Is64BitProcess) {
        # + Registries that are made in the Wow6432 Hive
        $kReaderLockDownWowHive = "HKLM:\Software\Wow6432Node\Policies\Adobe\Acrobat Reader\$ChannelName\FeatureLockDown"
        If(Test-Path $kReaderLockDownWowHive) {
            PrintAndLog "DEBUG: [Wow6432] - FeatureLockDown Key available"
            $objReaderWowKey = Get-Item -LiteralPath $kReaderLockDownWowHive
            If ($objReaderWowKey.GetValue("bUpdater", $null) -ne $null) {
                PrintAndLog "DEBUG: [Wow6432] - bUpdater Key present"
                Remove-ItemProperty -Path $kReaderLockDownWowHive -Name bUpdater -Force
                PrintAndLog "DEBUG: [Wow6432] - bUpdater Key removed"
            } Else {
                PrintAndLog "DEBUG: [Wow6432] - bUpdater Key not present"
            }
        } Else {
            PrintAndLog "DEBUG: [Wow6432] - FeatureLockDown Key not present"
        }
    }
}

function RemoveNotificationKey
{
##---------------------------------------------------------------------------------
## Function: RemoveNotificationKey
## Purpose: This function removes the Notification key in App-V Acrobat Reader
##---------------------------------------------------------------------------------
    PrintAndLog "-------------------------------------------------------------------------------"
    PrintAndLog "Removing Policies key to shutdown cServices\bToggleNotifications"
    PrintAndLog "-------------------------------------------------------------------------------"
            
    # Registries that are made in the native hive
    $kReaderLockDownHive = "HKLM:\Software\Policies\Adobe\Acrobat Reader\$ChannelName\FeatureLockDown\cServices"
    If(Test-Path $kReaderLockDownHive) {
        PrintAndLog "DEBUG: [Native] - cServices FeatureLockDown Key available"
        $objReaderKey = Get-Item -LiteralPath $kReaderLockDownHive
        If ($objReaderKey.GetValue("bToggleNotifications", $null) -ne $null) {
            PrintAndLog "DEBUG: [Native] - bToggleNotifications Key present"
            Remove-ItemProperty -Path $kReaderLockDownHive -Name bToggleNotifications -Force
            PrintAndLog "DEBUG: [Native] - bToggleNotifications Key removed"
        } Else {
            PrintAndLog "DEBUG: [Native] - bToggleNotifications Key not present"
        }
    } Else {
        PrintAndLog "DEBUG: [Native] - cServices FeatureLockDown Key not present"
    }
        
    If([System.Environment]::Is64BitProcess) {
        # + Registries that are made in the Wow6432 Hive
        $kReaderLockDownWowHive = "HKLM:\Software\Wow6432Node\Policies\Adobe\Acrobat Reader\$ChannelName\FeatureLockDown\cServices"
        If(Test-Path $kReaderLockDownWowHive) {
            PrintAndLog "DEBUG: [Wow6432] - cServices FeatureLockDown Key available"
            $objReaderWowKey = Get-Item -LiteralPath $kReaderLockDownWowHive
            If ($objReaderWowKey.GetValue("bToggleNotifications", $null) -ne $null) {
                PrintAndLog "DEBUG: [Wow6432] - bToggleNotifications Key present"
                Remove-ItemProperty -Path $kReaderLockDownWowHive -Name bToggleNotifications -Force
                PrintAndLog "DEBUG: [Wow6432] - bToggleNotifications Key removed"
            } Else {
                PrintAndLog "DEBUG: [Wow6432] - bToggleNotifications Key not present"
            }
        } Else {
            PrintAndLog "DEBUG: [Wow6432] - cServices FeatureLockDown Key not present"
        }
    }
}
 
function RemoveAppVRegistry
{
##---------------------------------------------------------------------------------
## Function: RemoveAppVRegistry
## Purpose: This function is used to remove appv key created at publish time
##---------------------------------------------------------------------------------
 
    PrintAndLog "-------------------------------------------------------------------------------"
    PrintAndLog "Removing appv registry key"
    PrintAndLog "-------------------------------------------------------------------------------"
    # Registries that are made in Native Hive
    $kAcrobatAppVKey = "HKLM:\Software\Adobe\Acrobat Reader\$ChannelName\appv"
    If (Test-Path $kAcrobatAppvKey){
        PrintAndLog "DEBUG: [Native] - Removing $kAcrobatAppVKey"
        Remove-Item $kAcrobatAppVKey
    } 
    # + Registries if made in Wow Hive
    If([System.Environment]::Is64BitProcess){
        $kAcrobatWow64AppVHive = "HKLM:\Software\Wow6432Node\Adobe\Acrobat Reader\$ChannelName\appv"
        If (Test-Path $kAcrobatWow64AppvHive) {
            PrintAndLog "DEBUG: [Wow6432] - $kAcrobatWow64AppVHive is available"
            Remove-Item $kAcrobatWow64AppVHive
        }
    }
}
 
function Usage
{
##-----------------------------------------------------------------------
## Function: Usage
## Purpose: This function is used to describe how to use this script
##-----------------------------------------------------------------------
PrintAndLog
@"
    Usage:
         
     .\Unpublish.ps1 [-CreateLog] [-LogFile <path to logfile>] [-ChannelName {"DC|2015|2017"}]
     Runs the Unpublish script to remove the dependencies of Adobe Acrobat Reader DC created as a part of publishing the package to the machine.
     e.g. .\Unpublish.ps1 -CreateLog -LogFile testing.log -ChannelName "2015|2017"
         
"@
    return
}
 
# +----------------------------------------------------------------------------
# | MAIN ROUTINE
# +----------------------------------------------------------------------------
function Main
{
    ShowHeading
    RemoveUpdaterKey
    RemoveNotificationKey
    RemoveAppVRegistry
    PrintAndLog "-------------------------------------------------------------------------------"
}
Main